package hbase;

import java.io.IOException;

import org.apache.hadoop.hbase.HBaseConfiguration;
import org.apache.hadoop.hbase.KeyValue;
import org.apache.hadoop.hbase.client.Result;
import org.apache.hadoop.hbase.client.Scan;
import org.apache.hadoop.hbase.filter.FirstKeyOnlyFilter;
import org.apache.hadoop.hbase.io.ImmutableBytesWritable;
import org.apache.hadoop.hbase.mapreduce.TableMapReduceUtil;
import org.apache.hadoop.hbase.mapreduce.TableMapper;
import org.apache.hadoop.mapreduce.Job;
import org.apache.hadoop.mapreduce.lib.output.NullOutputFormat;

public class RowCounter {
	static class RowCounterMapper extends TableMapper<ImmutableBytesWritable, Result> {
		/** Counter enumeration to count the actual rows. */
		public static enum Counters {
			ROWS
		}

		@Override
		public void map(ImmutableBytesWritable row, Result values, Context context) throws IOException {
			for (KeyValue value : values.list()) {
				if (value.getValue().length > 0) {
					context.getCounter(Counters.ROWS).increment(1);
					break;
				}
			}
		}
	}

	public static void main(String[] args) throws Exception {
		if(args.length != 1) {
			System.err.println("Usage: RowCounter <tablename>");
			System.exit(-1);
		}
		String tableName = args[0];
		
		Job job = new Job(HBaseConfiguration.create(), "rowcounter "+tableName);
		job.setJarByClass(RowCounter.class);
		job.setOutputFormatClass(NullOutputFormat.class);
		Scan scan = new Scan();
		scan.setFilter(new FirstKeyOnlyFilter());
		TableMapReduceUtil.initTableMapperJob(
				tableName,						// input table
				scan,							// input scan instance
				RowCounterMapper.class,			// mapper class
				ImmutableBytesWritable.class,	// type of output key
				Result.class,					// type of output value
				job);
		job.setNumReduceTasks(0);
		
		System.exit(job.waitForCompletion(true) ? 0 : 1);
	}
}
