/************************************************************/
//              DESCRIPTION
// 
// 
// Simple Scenario with one PWR UOX and one PWR MOX
// 
//                                   
//   _______     ____    _______    ___________
//  |       |   |    |  |       |  |	       |
//  |Reactor| =>|Pool|=>|Storage|=>|Fabrication| 
//  | UOx   |	|UOX |	| UOX 	|  |   Plant   |
//  |_______|   |____|  |_______|  |___________|
//  				      	||
//				      	\/
//               _______     ____     _______
//              |       |   |    |   |       |
//              |Storage|<= |Pool|<==|Reactor|
//              | MOX   |   |MOX |   | MOX   |
//              |_______|   |____|   |_______|
//
//
//@author FrA
/***********************************************************/
#include "CLASSHeaders.hxx"
#include <sstream>
#include <iomanip>
#include <math.h>
#include <string>
#include "XS/XSM_MLP.hxx"			//Load the include for Neural network cross section predictor
#include "Irradiation/IM_RK4.hxx"		//Load the include for Runge Kutta 4 resolution
#include "Equivalence/EQM_PWR_MLP_MOX.hxx"	//Load the include for Neural Network Equivalence Model (PWRMOX)
using namespace std;

int main(int argc, char** argv)
{
  //seconds in one year
  cSecond year = 3600*24.*365.25; 


  /******LOG MANAGEMENT**********************************/

  //Definition of the Log file : CLASS messages output 
  int Std_output_level 	= 0; // Only error are shown in terminal
  int File_output_level = 2; // Error + Warning + Info are shown in the file CLASS_OUTPUT.log
  CLASSLogger *Logger 	= new CLASSLogger("CLASS_OUTPUT.log",Std_output_level,File_output_level);


  /******SCENARIO****************************************/
  
  // The scenario start at year 2015
  Scenario *gCLASS=new Scenario(2015*year,Logger);
  gCLASS->SetStockManagement(true); //If false all the IsotopicVector in stocks are mixed together.
  gCLASS->SetTimeStep(year/4.); //the scenario calculation is updated every 3 months
  cSecond EndOfScenarioTime=2060*year; //Scenario ends in year 2060
  gCLASS->SetOutputFileName("Scenario6.root");//Set the name of the output file


  /******DATA BASES**************************************/

  //Geting CLASS to path
  string CLASS_PATH = getenv("CLASS_PATH");
  if (CLASS_PATH=="")
    {
      cout<<" Please setenv CLASS_PATH to your CLASS installation folder in your .bashs or .tcshrc"<<endl;
      exit(0);
    }
  string PATH_TO_DATA = CLASS_PATH + "/DATA_BASES/";

  /*===Decay data base===*/

  //The decay data base is taken from the file Decay.idx
  DecayDataBank* DecayDB = new DecayDataBank(gCLASS->GetLog(), PATH_TO_DATA + "DECAY/ALL/Decay.idx"); 
  //you may have to open this file and do the proper changes according your path
  gCLASS->SetDecayDataBase(DecayDB);
  //This decay data base will be used for all the decay calculations in this Scenario

  /*===Reactor data base===*/

  // Reprocessed fuel PWR MOX
  XSM_MLP* XSMOX = new XSM_MLP(gCLASS->GetLog(), PATH_TO_DATA + "PWR/MOX/XSModel/30Wg_FullMOX");
  //Defining the XS Predictor
  IM_RK4 *IMRK4 = new IM_RK4(gCLASS->GetLog());	//Bateman's equation solver method (RungeKutta4)

                                         //NO PATH FOR THE RK SOLVER???

  EQM_PWR_MLP_MOX* EQMMLPPWRMOX = new EQM_PWR_MLP_MOX(gCLASS->GetLog(),PATH_TO_DATA + "PWR/MOX/EQModel/MLP/EQM_MLP_PWR_MOX_3batch.xml");//Defining the EquivalenceModel
  PhysicsModels* PHYMOD = new PhysicsModels(XSMOX, EQMMLPPWRMOX, IMRK4);//The PhysicsModels containing the 3 object previously defined

  //Fixed fuel : PWR UOX
  EvolutionData *CYCLADE =new EvolutionData(gCLASS->GetLog(), PATH_TO_DATA + "PWR/UOX/FixedFuel/CYCLADES.dat");
  EvolutionData *GARANCE = new EvolutionData(gCLASS->GetLog(), PATH_TO_DATA + "PWR/UOX/FixedFuel/GARANCE.dat");
  EvolutionData *STD900 = new EvolutionData(gCLASS->GetLog(), PATH_TO_DATA + "PWR/UOX/FixedFuel/STD900.dat");


  /******FACILITIES*************************************/

  /*=== Stock===*/

  //Storage for infinite U and Pu
  Storage *Stock = new Storage(gCLASS->GetLog());    //Definition of the stock
  Stock->SetName("Stock");   //Its name
  Stock->AddToStock(ZAI(92,238,0) *  3e31);
  Stock->AddToStock(ZAI(92,235,0) * 0.04375* 3e31);
  /*  Stock->AddToStock(ZAI(94,238,0) *  10e28);
  Stock->AddToStock(ZAI(94,239,0) *  10e28);
  Stock->AddToStock(ZAI(94,240,0) *  10e28);
  Stock->AddToStock(ZAI(94,241,0) *  10e28);
  Stock->AddToStock(ZAI(94,242,0) *  10e28);*/

  gCLASS->Add(Stock);	//Adding the stock to the Scenario
  
  //Storage for MOX
  Storage *StockMOX = new Storage(gCLASS->GetLog());//Definition of the stock
  StockMOX->SetName("StockMOX");       	//Its name
  gCLASS->Add(StockMOX);	      	//Adding the stock to the Scenario

  //Storage for UOX
  Storage *StockUOX = new Storage(gCLASS->GetLog());//Definition of the stock
  StockUOX->SetName("StockUOX");	//Its name
  gCLASS->Add(StockUOX);	       	//Adding the stock to the Scenario

  /*===Pool===*/

  //Pool for UOX
  Pool *Cooling_UOX = new Pool(gCLASS->GetLog(),StockUOX, 5*year); //After 5 years of cooling, the pool sends its content to "StockUOX"
  Cooling_UOX->SetName("Pool_UOX");
  gCLASS->Add(Cooling_UOX);

  //Pool for MOX
  Pool *Cooling_MOX = new Pool(gCLASS->GetLog(),StockMOX, 5*year); //After 5 years of cooling, the pool sends its content to "StockMOX"
  Cooling_MOX->SetName("Pool_MOX");
  gCLASS->Add(Cooling_MOX);

  /*===A FabricationPlant===*/
  FabricationPlant *FP_MOX = new FabricationPlant(gCLASS->GetLog(), 3*year); //Declare a FabricationPlant. After the build of the fuel, it decays during 3years before to be loaded in Reactor
  FP_MOX->SetFiFo(false); //The latest isotopicVector to enter in "Stock" will be used to build the fuel (Opposite of First In First Out)
  FP_MOX->SetName("Fab_MOX");
  FP_MOX->AddFissileStorage(StockUOX);	//Tell the FP to look in Stock for fissionable material 
  FP_MOX->AddFertileStorage(Stock);//Tell the FP to look in Stock for fertile material 
  //If fertile stock is not defined (like here), CLASS get fertile from nature (OUTCOMING vector)
  //FP_MOX->SetReUsableStorage(wastestock);//By default the fabricationplant get the list of nuclei defined in the EquivalenceModel (here EQM_MLP_MOX) from stock and send the others nuclei in WASTE. If user want these nuclei to go in another stock  he can use the SetReUsableStorage function
  gCLASS->AddFabricationPlant(FP_MOX);


  /*=== Reactor===*/
   double	Power_CP0 = 2.66e10;	    //Thermal power (in W)
  //Combustibles type : CYCLADE
  double  BurnUp_Cyclade    = 47; 		// GWd/tHM
  double  HMMass_Cyclade    = 72.3;		//heavy metal mass (in tons)
	
  //REACTOR PWR UOX
  cSecond StartingTime =  2016*year;
  cSecond LifeTime     =  (EndOfScenarioTime - StartingTime);
  Reactor* PWR_UOX = new Reactor(gCLASS->GetLog(),// Log
   CYCLADE,	// The DataBase used
   Cooling_UOX,	// Connected Backend facility : The reactor discharge its fuel into the Pool "Cooling_UOX"
   StartingTime,// Starting time
   LifeTime,	// time of reactor life time
   Power_CP0,	// Power
   HMMass_Cyclade,// HM mass
   BurnUp_Cyclade,// BurnUp
   0.8);	// Load Factor
							
  PWR_UOX->SetName("PWR_UOX");// name of the reactor (as it will show up in the CLASSGui)
  gCLASS->AddReactor(PWR_UOX);//Add this reactor to the scenario

  //POWER reactor MOX
  double Power_CPY = 2.785e9;	    //Thermal power (in W)

  //Combustibles type : STANDARD 900
  double  BurnUpMOX = 35;             //GWd/tHM
  //double  BurnUp_STD900    = 33;      // GWd/tHM
  double  HMMass_STD900    = 72.3;    //heavy metal mass (in tons)
								
			
  //the PWR MOX
  StartingTime = 2022*year;
  LifeTime     =  EndOfScenarioTime - StartingTime;
  Reactor* PWR_MOX = new Reactor(gCLASS->GetLog(),// Log
   PHYMOD,			// The models used to build the fuel & to calculate its evolution
   FP_MOX,			// The FabricationPlant
   Cooling_MOX,			// Connected Backend
   StartingTime,		// Starting time
   LifeTime,			// time of reactor l
   Power_CPY,			// Power
   HMMass_STD900,		// HM mass
   BurnUpMOX,			// BurnUp
   0.8);			// Load Factor

  PWR_MOX->SetName("PWR_MOX");// name of the reactor (as it will appear in the CLASSGui)
  gCLASS->AddReactor(PWR_MOX);//Add this reactor to the scenario


  gCLASS->Evolution((double)EndOfScenarioTime);//Perform the calculation from year 2015(defined in Scenario declaration) to year 2060

	delete gCLASS;

}


//==========================================================================================
// Compilation
//==========================================================================================
/*
 
 \rm CLASS* ; g++ -o CLASS_Exec Scenario6.cxx -I $CLASS_include -L $CLASS_lib -lCLASSpkg `root-config --cflags` `root-config --libs` -fopenmp -lgomp -Wunused-result
 
 
 */
